﻿namespace Microsoft.Samples.PlanMyNight.Web.Controllers.Tests
{
    using System.Collections.Specialized;
    using System.Web.Mvc;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.Samples.PlanMyNight.Web.Controllers;
    using Microsoft.Samples.PlanMyNight.Web.ViewModels;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Moq;
    using MvcContrib.TestHelper;
    using Rhino.Mocks;
    using WindowsLiveId;

    [TestClass]
    public class AccountControllerFixture
    {
        [TestMethod]
        public void Index()
        {
            var repository = new Mock<IActivitiesRepository>();
            repository.Setup(o => o.RetrieveActivityTypes()).Returns(new[] { new ActivityType { Id = 1, Name = "Bar" }, new ActivityType { Id = 2, Name = "Restaurant" } });

            var reference = new Mock<IReferenceRepository>();
            reference.Setup(o => o.RetrieveStates()).Returns(new[] { new State { Name = "Ohio", Abbreviation = "OH" } });

            var membership = new Mock<IMembershipService>();
            membership.Setup(o => o.GetCurrentProfile()).Returns(new UserProfile() { FullName = "John Doe", City = "New York", State = "NY" });

            var builder = new TestControllerBuilder();
            var controller = new AccountController(membership.Object, new Mock<IWindowsLiveLogin>().Object, new Mock<IFormsAuthentication>().Object, reference.Object, repository.Object);
            builder.InitializeController(controller);

            var result = controller.Index("~/test-path/");
            var viewData = result.AssertViewRendered().ViewData;

            var model = viewData.Model as ProfileViewModel;
            Assert.AreEqual("John Doe", model.Profile.FullName);
            Assert.AreEqual("New York", model.Profile.City);
            Assert.IsNotNull(model.PreferredActivityTypes);
            Assert.IsNotNull(model.States);
            Assert.AreEqual("~/test-path/", model.ReturnUrl);
            repository.VerifyAll();
        }

        [TestMethod]
        public void Update()
        {
            var membership = new Mock<IMembershipService>();
            membership.Setup(o => o.UpdateProfile(It.IsAny<UserProfile>()));

            var builder = new TestControllerBuilder();
            var userProfile = new UserProfile()
            {
                FullName = "John Doe Updated",
                State = string.Empty,
                City = "Columbus",
                PreferredActivityTypeId = 1
            };

            var controller = new AccountController(membership.Object, new Mock<IWindowsLiveLogin>().Object, new Mock<IFormsAuthentication>().Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            controller.Update(userProfile).AssertResultIs<RedirectToRouteResult>();

            membership.Verify(o => o.UpdateProfile(It.Is<UserProfile>(u => u == userProfile)));
        }

        [TestMethod]
        public void UpdateSuccess()
        {
            var builder = new TestControllerBuilder();
            var controller = new AccountController(new Mock<IMembershipService>().Object, new Mock<IWindowsLiveLogin>().Object, new Mock<IFormsAuthentication>().Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            controller.UpdateSuccess(null).AssertViewRendered();
        }

        [TestMethod]
        public void LoginRedirectsToLiveId()
        {
            var login = new Mock<IWindowsLiveLogin>();
            login.Setup(o => o.GetLoginUrl(It.IsAny<string>()))
                 .Returns("http://loginsite/");

            var browserMock = new Mock<System.Web.HttpBrowserCapabilitiesBase>();
            browserMock.Setup(o => o.IsMobileDevice).Returns(false);

            var builder = new TestControllerBuilder();
            builder.HttpContext.Request.Stub(r => r.Browser).Return(browserMock.Object);
            var controller = new AccountController(new Mock<IMembershipService>().Object, login.Object, new Mock<IFormsAuthentication>().Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            var url = controller.Login(string.Empty).AssertHttpRedirect().Url;
            Assert.AreEqual("http://loginsite/", url);
        }

        [TestMethod]
        public void LoginRedirectsToMobileLiveId()
        {
            var login = new Mock<IWindowsLiveLogin>();
            login.Setup(o => o.GetMobileLoginUrl(It.IsAny<string>()))
                 .Returns("http://mobileloginsite/");

            var browserMock = new Mock<System.Web.HttpBrowserCapabilitiesBase>();
            browserMock.Setup(o => o.IsMobileDevice).Returns(true);

            var builder = new TestControllerBuilder();
            builder.HttpContext.Request.Stub(r => r.Browser).Return(browserMock.Object);
            var controller = new AccountController(new Mock<IMembershipService>().Object, login.Object, new Mock<IFormsAuthentication>().Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            var url = controller.Login(string.Empty).AssertHttpRedirect().Url;
            Assert.AreEqual("http://mobileloginsite/", url);
        }

        [TestMethod]
        public void LiveIdLogoutDeletesCookiesAndRedirectsToHome()
        {
            var login = new Mock<IWindowsLiveLogin>();
            var authentication = new Mock<IFormsAuthentication>();
            authentication.Setup(o => o.SignOut());

            var builder = new TestControllerBuilder();
            builder.QueryString["action"] = "logout";

            var controller = new AccountController(new Mock<IMembershipService>().Object, login.Object, authentication.Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            var url = controller.LiveId().AssertHttpRedirect().ToUrl("~/");
            authentication.VerifyAll();
        }

        [TestMethod]
        public void LiveIdClearCookieDeletesCookiesAndReturnsIcon()
        {
            var login = new Mock<IWindowsLiveLogin>();
            string type = "image/gif";
            byte[] data = new byte[0];
            login.Setup(o => o.GetClearCookieResponse(out type, out data));

            var authentication = new Mock<IFormsAuthentication>();
            authentication.Setup(o => o.SignOut());

            var builder = new TestControllerBuilder();
            builder.QueryString["action"] = "clearcookie";

            var controller = new AccountController(new Mock<IMembershipService>().Object, login.Object, authentication.Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            controller.LiveId().AssertResultIs<FileStreamResult>();
            authentication.VerifyAll();
            login.VerifyAll();
        }

        [TestMethod]
        public void LiveIdFirstLoginRegistersUserAnthenticatesAndRedirectsToProfile()
        {
            var login = new Mock<IWindowsLiveLogin>();
            login.Setup(o => o.ProcessLogin(It.IsAny<NameValueCollection>()))
                 .Returns(new WindowsLiveLogin.User("1000", "bd269fd1bde84396ac739a7a7434f673", string.Empty, string.Empty, string.Empty));

            var membership = new Mock<IMembershipService>();
            membership.Setup(o => o.ValidateUser("bd269fd1-bde8-4396-ac73-9a7a7434f673", "bd269fd1-bde8-4396-ac73-9a7a7434f673"))
                      .Returns(false);
            membership.Setup(o => o.CreateUser("bd269fd1-bde8-4396-ac73-9a7a7434f673", "bd269fd1-bde8-4396-ac73-9a7a7434f673", string.Empty))
                      .Returns(System.Web.Security.MembershipCreateStatus.Success);
            membership.Setup(o => o.CreateProfile("bd269fd1-bde8-4396-ac73-9a7a7434f673")).Returns(new UserProfile());
            membership.Setup(o => o.UpdateProfile(It.IsAny<UserProfile>()));

            var authentication = new Mock<IFormsAuthentication>();
            authentication.Setup(o => o.SignIn("bd269fd1-bde8-4396-ac73-9a7a7434f673", false));

            var builder = new TestControllerBuilder();
            builder.QueryString["action"] = "login";

            var controller = new AccountController(membership.Object, login.Object, authentication.Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            controller.LiveId().AssertActionRedirect().ToAction("Index");
            login.VerifyAll();
            authentication.VerifyAll();
            membership.VerifyAll();
        }

        [TestMethod]
        public void LiveIdSecondLoginAuthenticatesAndRedirectsToReturnUrl()
        {
            var login = new Mock<IWindowsLiveLogin>();
            login.Setup(o => o.ProcessLogin(It.IsAny<NameValueCollection>()))
                 .Returns(new WindowsLiveLogin.User("1000", "bd269fd1bde84396ac739a7a7434f673", string.Empty, "/Search/", string.Empty));

            var membership = new Mock<IMembershipService>();
            membership.Setup(o => o.ValidateUser("bd269fd1-bde8-4396-ac73-9a7a7434f673", "bd269fd1-bde8-4396-ac73-9a7a7434f673"))
                      .Returns(true);

            var authentication = new Mock<IFormsAuthentication>();
            authentication.Setup(o => o.SignIn("bd269fd1-bde8-4396-ac73-9a7a7434f673", false));

            var builder = new TestControllerBuilder();
            builder.QueryString["action"] = "login";

            var controller = new AccountController(membership.Object, login.Object, authentication.Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            controller.LiveId().AssertHttpRedirect().ToUrl("/Search/");
            login.VerifyAll();
            authentication.VerifyAll();
            membership.VerifyAll();
        }

        [TestMethod]
        public void LiveIdSecondLoginAuthenticatesAndRedirectsToHome()
        {
            var login = new Mock<IWindowsLiveLogin>();
            login.Setup(o => o.ProcessLogin(It.IsAny<NameValueCollection>()))
                 .Returns(new WindowsLiveLogin.User("1000", "bd269fd1bde84396ac739a7a7434f673", string.Empty, string.Empty, string.Empty));

            var membership = new Mock<IMembershipService>();
            membership.Setup(o => o.ValidateUser("bd269fd1-bde8-4396-ac73-9a7a7434f673", "bd269fd1-bde8-4396-ac73-9a7a7434f673"))
                      .Returns(true);

            var authentication = new Mock<IFormsAuthentication>();
            authentication.Setup(o => o.SignIn("bd269fd1-bde8-4396-ac73-9a7a7434f673", false));

            var builder = new TestControllerBuilder();
            builder.QueryString["action"] = "login";

            var controller = new AccountController(membership.Object, login.Object, authentication.Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            controller.LiveId().AssertHttpRedirect().ToUrl("~/");
            login.VerifyAll();
            authentication.VerifyAll();
            membership.VerifyAll();
        }

        [TestMethod]
        public void LiveIdWrongTokenRedirectsToHome()
        {
            var login = new Mock<IWindowsLiveLogin>();
            login.Setup(o => o.ProcessLogin(It.IsAny<NameValueCollection>()))
                 .Returns(default(WindowsLiveLogin.User));

            var membership = new Mock<IMembershipService>();

            var authentication = new Mock<IFormsAuthentication>();

            var builder = new TestControllerBuilder();
            builder.QueryString["action"] = "login";

            var controller = new AccountController(membership.Object, login.Object, authentication.Object, new Mock<IReferenceRepository>().Object, new Mock<IActivitiesRepository>().Object);
            builder.InitializeController(controller);

            controller.LiveId().AssertHttpRedirect().ToUrl("~/");
            login.VerifyAll();
        }
    }
}
